/*
 * Utility methods to extract information from INVENTORY response.
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"

/*
 *	Segment-specific information
 */

/** Return true if @seg has at least a processor or a memory allocation. */
static bool segment_is_allocated(const struct basil_segment *seg)
{
	struct basil_node_processor *proc;
	struct basil_node_memory *mem;

	for (proc = seg->proc_head; proc; proc = proc->next)
		if (proc->allocation != NULL)
			return true;

	for (mem = seg->mem_head; mem; mem = mem->next)
		if (mem->a_head != NULL)
			return true;
	return false;
}

/*
 *	CPU/core-specific information
 */

/**
 * node_is_allocated  -  return true if @node is claimed by a reservation
 * A node is allocated if it has at least 1 processor or 1 memory allocation.
 */
bool node_is_allocated(const struct basil_node *node)
{
	struct basil_segment *seg;

	for (seg = node->seg_head; seg; seg = seg->next)
		if (segment_is_allocated(seg))
			return true;
	return false;
}

/**
 * get_free_nodes - perform INVENTORY and return list of available nodes
 */
struct nodespec *get_free_nodes(enum basil_version version)
{
	struct nodespec *head = NULL;
	struct basil_node *node;
	struct basil_inventory *inv;

	inv = get_full_inventory(version);
	if (inv == NULL)
		return NULL;

	/* Use the same rule as the scheduler to classify a node as free */
	for (node = inv->f->node_head; node; node = node->next) {
		if (node->state != BNS_UP    ||
		    node->role  != BNR_BATCH ||
		    node_is_allocated(node))
			continue;

		if (ns_add_range(&head, node->node_id, node->node_id) != 0) {
			error("Can not perform node inventory");
			free_nodespec(head);
			head = NULL;
			break;
		}
	}
	free_inv(inv);
	return head;
}

/**
 * basil_node_total_count  -  return total node count in @inv
 */
uint32_t basil_node_total_count(const struct basil_inventory *inv)
{
	struct basil_node *node;
	uint32_t node_count = 0;

	if (inv && inv->f)
		for (node = inv->f->node_head; node; node = node->next)
			node_count++;
	return node_count;
}

/**
 * basil_node_total_cores  -  total number of processors available on @node
 */
uint16_t basil_node_total_cores(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_processor *proc;
	uint16_t total_processors = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (proc = seg->proc_head; proc; proc = proc->next)
			total_processors++;
	return total_processors;
}

/**
 * basil_node_reserved_cores  -  number of reserved processors on @node
 */
uint16_t basil_node_reserved_cores(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_processor *proc;
	uint16_t reserved_processors = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (proc = seg->proc_head; proc; proc = proc->next)
			if (proc->allocation)
				reserved_processors++;
	return reserved_processors;
}


/*
 *	Memory-specific information
 */

/**
 * basil_node_page_size  -  (homogeneous) page size on @node in KB
 */
uint32_t basil_node_page_size(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_memory *mem;
	uint32_t page_size_kb = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (mem = seg->mem_head; mem; mem = mem->next) {
			if (page_size_kb && page_size_kb != mem->page_size_kb)
				error("inhomogeneous node memory page size");
			page_size_kb = mem->page_size_kb;
		}
	return page_size_kb;
}

/**
 * basil_node_page_count  -  total page count on @node
 */
uint32_t basil_node_page_count(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_memory *mem;
	uint32_t page_count = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (mem = seg->mem_head; mem; mem = mem->next)
			page_count += mem->page_count;
	return page_count;
}

/**
 * basil_node_avail_mem  -  total memory available on @node
 */
uint64_t basil_node_avail_mem(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_memory *mem;
	uint64_t total_mem = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (mem = seg->mem_head; mem; mem = mem->next)
			total_mem += mem->page_size_kb * mem->page_count;
	return total_mem;
}


/**
 * basil_node_pages_conf  -  estimate the number of pages confirmed on @node
 */
uint32_t basil_node_pages_conf(const struct basil_node *node)
{
	struct basil_segment *seg;
	struct basil_node_memory *mem;
	struct basil_mem_alloc *ma;
	uint32_t reserved_page_count = 0;

	for (seg = node->seg_head; seg; seg = seg->next)
		for (mem = seg->mem_head; mem; mem = mem->next)
			for (ma = mem->a_head; ma; ma = ma->next)
				reserved_page_count += ma->page_count;
	return reserved_page_count;
}

/*
 *	Reservation/APID - specific Information
 */

/**
 * basil_rsvn_by_resid  -  Search for a particular reservation by @rsvn_id
 */
static const struct basil_rsvn *basil_rsvn_by_resid(const struct basil_rsvn *rsvn,
						    uint32_t rsvn_id)
{
	while (rsvn && rsvn->rsvn_id != rsvn_id)
		rsvn = rsvn->next;
	return rsvn;
}

/**
 * basil_rsvn_by_id  -  Search @inv for a particular reservation by @rsvn_id
 */
const struct basil_rsvn *basil_rsvn_by_id(const struct basil_inventory *inv,
					  uint32_t rsvn_id)
{
	assert(inv && inv->f);
	return basil_rsvn_by_resid(inv->f->rsvn_head, rsvn_id);
}

/**
 * basil_rsvn_params - return the reservation parameters of @rsvn
 */
static const struct basil_rsvn_app *basil_rsvn_params(const struct basil_rsvn *rsvn)
{
	const struct basil_rsvn_app *app;
	/*
	 * The reservation-specific entry has a timestamp of 0
	 * and a single Command entry with a 'cmd' of 'BASIL'.
	 */
	for (app = rsvn->app_head; app; app = app->next)
		if (app->timestamp == 0 &&
		    app->cmd_head && app->cmd_head->next == NULL &&
		    strcmp(app->cmd_head->cmd, "BASIL") == 0)
			return app;
	return NULL;
}

/**
 * basil_rsvn_params_by_id - return the reservation parameters of @rsvn_id
 */
const struct basil_rsvn_app *basil_rsvn_params_by_id(const struct basil_inventory *inv,
						     uint32_t rsvn_id)
{
	const struct basil_rsvn *rsvn = basil_rsvn_by_id(inv, rsvn_id);

	return rsvn ? basil_rsvn_params(rsvn) : NULL;
}

/**
 * basil_get_rsvn_apid  -  get the APID associated with @rsvn_id
 */
uint64_t basil_get_rsvn_apid(const struct basil_inventory *inv, uint32_t rsvn_id)
{
	const struct basil_rsvn_app *app = basil_rsvn_params_by_id(inv, rsvn_id);

	return app ? app->apid : 0;
}

/**
 * basil_rsvn_get_num_apps - number of applications associated with @rsvn
 * The first application (see comment below) refers to the reservation itself
 * and thus is not counted. Such a reservation is in state 'conf', if at least
 * 1 application is running, it is in state 'conf,claim'.
 */
static uint32_t basil_rsvn_get_num_apps(const struct basil_rsvn *rsvn)
{
	const struct basil_rsvn_app *app;
	uint32_t count = 0;
	/*
	 * basil(7) manpage:
	 * The first Application in each ApplicationArray describes the
	 * reservation itself. The time_stamp is set to zero and the cmd
	 * attribute is set to BASIL for this Application.
	 */
	for (app = rsvn->app_head; app; app = app->next)
		if (app->timestamp)
			count++;
	return count;
}

/**
 * basil_rsvn_get_num_apps_by_id - number of applications associated with @rsvn_id
 */
uint32_t basil_rsvn_get_num_apps_by_id(const struct basil_inventory *inv,
				       uint32_t rsvn_id)
{
	const struct basil_rsvn *rsvn = basil_rsvn_by_id(inv, rsvn_id);

	return rsvn ? basil_rsvn_get_num_apps(rsvn) : 0;
}

/**
 * basil_get_rsvn_aprun_apids  -  get list of aprun APIDs for @rsvn_id
 * Returns 0-terminated array, which caller must free.
 * WARNING: if the aprun application uses fewer nodes than are reserved by
 *          @rsvn_id, additional information is required to confirm whether
 *          that particular node is indeed in use by the given apid.
 */
uint64_t *basil_get_rsvn_aprun_apids(const struct basil_inventory *inv,
				     uint32_t rsvn_id)
{
	const struct basil_rsvn	*rsvn = basil_rsvn_by_id(inv, rsvn_id);
	const struct basil_rsvn_app *app;
	uint64_t *apids = NULL;
	int n = 1;	/* 0-terminated array */

	if (rsvn == NULL)
		return NULL;

	for (app = rsvn->app_head; app; app = app->next)
		/*
		 * There are two types of basil_rsvn_app applications:
		 * - the first application has a 'timestamp' of 0, a 'cmd' of
		 *   "BASIL" - this is used to store the reservation parameters;
		 * - all other applications have a non-0 timestamp and refer to
		 *   actual aprun job steps (whose APIDs we are interested in).
		 */
		if (app->timestamp) {
			apids = realloc(apids, (n + 1) * sizeof(*apids));
			if (apids == NULL)
				fatal("failed to allocate Apid entry");
			apids[n-1] = app->apid;
			apids[n++] = 0;
		}
	return apids;
}

/**
 * basil_rsvn_list_push  -  push unique reservation Id onto the list
 * @head:	start of list (stack order)
 * @resv_id:	reservation ID to push onto the top of the stack
 * This only uses the rsvn_id to create a stack of IDs, the other parameters
 * and nested structs remain unused.
 */
static void basil_rsvn_list_push(struct basil_rsvn **head, uint32_t rsvn_id)
{
	struct basil_rsvn *cur;

	assert(head != NULL);
	for (cur = *head; cur; cur = cur->next)
		if (cur->rsvn_id == rsvn_id)
			return;

	cur = calloc(sizeof(*cur), 1);
	if (cur == NULL)
		fatal("can not allocate basil_rsvn_list");

	cur->rsvn_id = rsvn_id;
	cur->next    = *head;
	*head = cur;
}

/**
 * basil_rsvns_on_node  -  extract list of unique reservation IDs claiming @node
 */
struct basil_rsvn *basil_rsvns_on_node(const struct basil_node *node)
{
	struct basil_rsvn *resid_stack = NULL;
	const struct basil_segment *seg;
	const struct basil_node_processor *proc;
	const struct basil_node_memory *mem;
	const struct basil_mem_alloc *ma;

	for (seg = node->seg_head; seg; seg = seg->next) {
		for (proc = seg->proc_head; proc; proc = proc->next)
			if (proc->allocation)
				basil_rsvn_list_push(&resid_stack,
						     proc->allocation->rsvn_id);
		for (mem = seg->mem_head; mem; mem = mem->next)
			for (ma = mem->a_head; ma; ma = ma->next)
				basil_rsvn_list_push(&resid_stack, ma->rsvn_id);
	}
	return resid_stack;
}
