/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

/////////////////////// stdlib includes


/////////////////////// Qt includes
#include <QDebug>


/////////////////////// pappsomspp includes


/////////////////////// libXpertMassGui includes


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/Tolerance.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{

Tolerance::Tolerance(QObject *parent): QObject(parent)
{
}

Tolerance::Tolerance(Tolerance::Type type, double nominal, QObject *parent)
  : QObject(parent), m_nominal(nominal), m_type(type)
{
}

void
Tolerance::setNominal(double nominal)
{
  if(m_nominal == nominal)
    return;

  m_nominal = nominal;

  emit nominalChangedSignal();
  emit toleranceChangedSignal(
    std::pair<double, Tolerance::Type>(m_nominal, m_type));
}

double
Tolerance::getNominal() const
{
  return m_nominal;
}

void
Tolerance::setType(Type type)
{
  if(m_type == type)
    return;

  m_type = type;

  emit typeChangedSignal();
  emit toleranceChangedSignal(
    std::pair<double, Tolerance::Type>(m_nominal, m_type));
}

void
Tolerance::setType(const QString &text)
{
  std::map<Type, QString>::const_iterator the_iterator_cst =
    std::find_if(m_typeToStringMap.cbegin(),
                 m_typeToStringMap.cend(),
                 [&](const std::pair<Type, QString> &pair) {
                   if(pair.second == text)
                     return true;
                   return false;
                 });

  if(the_iterator_cst != m_typeToStringMap.cend())
    m_type = (*the_iterator_cst).first;
  else
    qFatalStream()
      << "Programming error. Tolerance type as string is not known.";

  emit typeChangedSignal();
  emit toleranceChangedSignal(
    std::pair<double, Tolerance::Type>(m_nominal, m_type));
}

Tolerance::Type
Tolerance::getType() const
{
  return m_type;
}

Tolerance::Type
Tolerance::getType(const QString &text) const
{
  std::map<Type, QString>::const_iterator the_iterator_cst =
    std::find_if(m_typeToStringMap.cbegin(),
                 m_typeToStringMap.cend(),
                 [&](const std::pair<Type, QString> &pair) {
                   if(pair.second == text)
                     return true;
                   return false;
                 });

  if(the_iterator_cst != m_typeToStringMap.cend())
    return (*the_iterator_cst).first;

  return Tolerance::Type::NOT_SET;
}

QString
Tolerance::getTypeAsString() const
{
  std::map<Type, QString>::const_iterator the_iterator_cst =
    std::find_if(m_typeToStringMap.cbegin(),
                 m_typeToStringMap.cend(),
                 [&](const std::pair<Type, QString> &pair) {
                   if(pair.first == m_type)
                     return true;
                   return false;
                 });

  if(the_iterator_cst != m_typeToStringMap.cend())
    return (*the_iterator_cst).second;

  return QString();
}

void
Tolerance::initialize(double nominal, Type type)
{
  m_nominal = nominal;
  m_type    = type;

  emit toleranceChangedSignal(
    std::pair<double, Tolerance::Type>(m_nominal, m_type));
}

void
Tolerance::initialize(const Tolerance &tolerance)
{
  initialize(tolerance.m_nominal, tolerance.m_type);
}

Tolerance *
Tolerance::clone(QObject *parent)
{
  Tolerance *copy = new Tolerance(m_type, m_nominal, parent);
  return copy;
}

double
Tolerance::calculateWidth(double reason) const
{
  double width = 0;

  if(m_type == Type::DALTON)
    width = m_nominal;
  else if(m_type == Type::RES)
    width = reason / m_nominal;
  else if(m_type == Type::PPM)
    width = reason / 1000000 * m_nominal;
  else
    qFatalStream() << "Programming error. The Type is not valid.";

  emit widthChangedSignal(width);

  return width;
}

QString
Tolerance::toString() const
{
  QString type_as_text = getTypeAsString();

  return QString("%1 %2").arg(m_nominal).arg(type_as_text);
}


} // namespace libXpertMassCore
} // namespace MsXpS
