"""
This script takes as input a CT scan that has been collected in "half-acquisition" mode
and produces a series of reconstructed slices, each of which are generated by cropping and
concatenating opposing projections together over a range of "overlap" values (i.e. the pixel column
at which the images are cropped and concatenated).
The objective is to review this series of images to determine the pixel column at which the axis of rotation
is located (much like the axis search function commonly used in reconstruction software).
"""

import os
import numpy as np
import tifffile

from tofu.ez.image_read_write import TiffSequenceReader
import tofu.ez.params as glob_parameters
from tofu.ez.Helpers.stitch_funcs import findCTdirs, stitch_float32_output



def extract_row(dir_name, row):
    tsr = TiffSequenceReader(dir_name)
    tmp = tsr.read(0)
    (N, M) = tmp.shape
    if (row < 0) or (row > N):
        row = N//2
    num_images = tsr.num_images
    if num_images % 2 == 1:
        print(f"odd number of images ({num_images}) in {dir_name}, "
              f"discarding the last one before stitching pairs")
        num_images-=1
    A = np.empty((num_images, M), dtype=np.uint16)
    for i in range(num_images):
        A[i, :] = tsr.read(i)[row, :]
    tsr.close()
    return A

def find_overlap(parameters):
    print("Finding CTDirs...")
    ctdirs, lvl0 = findCTdirs(parameters['360overlap_input_dir'],
                              glob_parameters.params['main_config_tomo_dir_name'])
    ctdirs.sort()
    print(ctdirs)

    # concatenate images end-to-end and generate a sinogram
    for ctset in ctdirs:
        print("Working on ctset:" + str(ctset))
        index_dir = os.path.basename(os.path.normpath(ctset))

        # loading:
        print(os.path.join(ctset, glob_parameters.params['main_config_flats_dir_name']))
        try:
            row_flat = np.mean(extract_row(
                os.path.join(ctset, glob_parameters.params['main_config_flats_dir_name']),
                           parameters['360overlap_row']))
        except:
            print(f"Problem loading flats in {ctset}")
            continue
        try:
            row_dark = np.mean(extract_row(
                os.path.join(ctset, glob_parameters.params['main_config_darks_dir_name']),
                                       parameters['360overlap_row']))
        except:
            print(f"Problem loading darks in {ctset}")
            continue
        try:
            row_tomo = extract_row(
                os.path.join(ctset, glob_parameters.params['main_config_tomo_dir_name']),
                                   parameters['360overlap_row'])
        except:
            print(f"Problem loading projections from "
                  f"{os.path.join(ctset, glob_parameters.params['main_config_tomo_dir_name'])}")
            continue

        row_flat2 = None
        tmpstr = os.path.join(ctset, glob_parameters.params['main_config_flats2_dir_name'])
        if os.path.exists(tmpstr):
            try:
                row_flat2 = np.mean(extract_row(tmpstr, parameters['360overlap_row']))
            except:
                print(f"Problem loading flats2 in {ctset}")


        (num_proj, M) = row_tomo.shape

        print('Flat-field correction...')
        # Flat-correction
        tmp_flat = np.tile(row_flat, (num_proj, 1))
        if row_flat2 is not None:
            tmp_flat2 = np.tile(row_flat2, (num_proj, 1))
            ramp = np.linspace(0, 1, num_proj)
            ramp = np.transpose(np.tile(ramp, (M, 1)))
            tmp_flat = tmp_flat * (1-ramp) + tmp_flat2 * ramp
            del ramp, tmp_flat2

        tmp_dark = np.tile(row_dark, (num_proj, 1))
        tomo_ffc = -np.log((row_tomo - tmp_dark)/np.float32(tmp_flat - tmp_dark))
        del row_tomo, row_dark, row_flat, tmp_flat, tmp_dark
        np.nan_to_num(tomo_ffc, copy=False, nan=0.0, posinf=0.0, neginf=0.0)

        # create interpolated sinogram of flats on the
        # same row as we use for the projections, then flat/dark correction
        print('Creating stitched sinograms...')

        sin_tmp_dir = os.path.join(parameters['360overlap_temp_dir'], index_dir)
        os.makedirs(sin_tmp_dir)
        for axis in range(parameters['360overlap_lower_limit'],
                          parameters['360overlap_upper_limit'],
                          parameters['360overlap_increment']):
            cro = parameters['360overlap_upper_limit'] - axis
            A = stitch_float32_output(
                tomo_ffc[: num_proj//2, :], tomo_ffc[num_proj//2:, ::-1], axis, cro)
            tifffile.imsave(os.path.join(
                sin_tmp_dir, 'sin-axis-' + str(axis).zfill(4) + '.tif'), A.astype(np.float32))

            # perform reconstructions for each sinogram and save to output folder

        print('Reconstructing slices...')
        reco_axis = M-parameters['360overlap_upper_limit']
        cmd = f'tofu tomo --axis {reco_axis} --sinograms {sin_tmp_dir}'
        cmd +=' --output '+os.path.join(os.path.join(
            parameters['360overlap_output_dir'], f"{index_dir}-sli.tif"))
        print(cmd)
        os.system(cmd)

        print("Finished processing: " + str(index_dir))
        print("********************DONE********************")

    #shutil.rmtree(parameters['360overlap_temp_dir'])
    print("Finished processing: " + str(parameters['360overlap_input_dir']))

