
//! @file apifunc.h
//! @ingroup InternalAPI
//! @brief Common internal helper functions
//!
//! This file declares some commonly used functions that implement api
//! features but which are not exposed to the application programmer.

/*---------------------------------------------------------------------------*\

    AUTHOR..: David Rowe
    DATE....: 4/2/98
    AUTHOR..: Ron Lee
    DATE....: 10/11/06


         Voicetronix Voice Processing Board (VPB) Software
         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#ifndef __APIFUNC__
#define __APIFUNC__

#include "comm.h"
#include "wobbly.h"


//! @defgroup InternalAPIfunc Miscellaneous functions
//! @ingroup InternalAPI
//! @brief A collection of internal implementation functions.
//!
//! These functions mostly mirror some of those from the public API but are
//! implemented without error checking, or for invocation using already
//! decomposed parameters.
//!@{
//{{{

//! Pointer to the global comm object
extern Comm *vpb_c;


int RunTimeError(const Wobbly &w, const char *api_function);

//! Check for a valid device handle.
//
//! The @a handle is valid if the device has been opened with a prior call
//! to @c vpb_open().
//!
//! @exception VpbException will be thrown, stating @a where, if @a handle is invalid.
void CheckHandle(VPBPortHandle handle, const char *where);

//XXX
void ValidHandleCheck(int handle);

//! Validate a @c NULL terminated string of @c DTMF @a digits.
//
//! @note this only includes the standard @c DTMF digits, user defined tone
//!       characters will fail if passed here, even if they are valid to be
//!       passed to @c vpb_dial_{a,}sync().
//! @exception Wobbly will be thrown if any of the digits are not valid.
void validate_digits(const char *digits);

//! Internal function to place an event on the API event queue.
//
//! @param event The event to place on the API queue.
//! @param mask	 The mask to use for this event (set to 0 if not required).
int putevt(VPB_EVENT *event, unsigned long mask = 0);

void apifunc_trace(char s[]);

//! Return the symbolic type of the driver for @a board.
int get_driver_type(int board);

//! Return the symbolic @c VPB_MODEL type of a @a board.
VPB_MODEL get_board_type(int board);

//! Return a descriptive model name for @a board in @a s.
void get_board_model(int board, char *s);

//! Return the type of a @a port on some @a board.  @c VPB_FXO or @c VPB_FXS.
VPB_PORT get_port_type(int board, int port);

//! Return the total number of ports on all registered boards.
size_t get_total_port_count();

//! Set the country data for @a port on @a board.
void set_country(int board, int port, const Country *country);

//! Return the country data for @a port on @a board.
const Country *get_country(int board, int port);

//! Internal function to control ringing on @a port of @a board.
void ring_station_async(int board, int port, int cadence);

//! Write directly to a @c V4PCI codec register.
//
//! @param handle The handle returned by a call to @c vpb_open().
//! @param addr   8-bit address of the codec register.
//! @param data   8-bit data to write to the register.
//! @param comm   The comm object used to send the message.
//!
//! Used to set a register of the TS5070 codecs, allows driver-level config
//! of codec, rather than hard-coding in DSP firmware.
void set_codec_reg(int handle, unsigned short addr, unsigned short data, Comm *comm);

//!@} //}}}  group InternalAPIfunc

#endif	/* #ifndef __APIFUNC__	*/

